#!/bin/bash
# ServerDash Installation Script

set -e

INSTALL_DIR="/opt/serverdash"
SERVICE_FILE="/etc/systemd/system/serverdash.service"

echo "Installing ServerDash..."

# Check if this is an upgrade (service already exists)
if systemctl is-active --quiet serverdash 2>/dev/null; then
    echo "Stopping existing ServerDash service for upgrade..."
    sudo systemctl stop serverdash
fi

# Create installation directory
sudo mkdir -p "${INSTALL_DIR}"

# Copy binary
sudo cp serverdash "${INSTALL_DIR}/"
sudo chmod +x "${INSTALL_DIR}/serverdash"

# Copy config (preserve existing if upgrading)
if [ -f "config.json" ]; then
    if [ -f "${INSTALL_DIR}/config.json" ]; then
        echo "Preserving existing config.json"
    else
        sudo cp config.json "${INSTALL_DIR}/"
    fi
fi

# Copy sample scripts (if included)
if [ -d "scripts" ]; then
    sudo cp -r scripts "${INSTALL_DIR}/"
    echo "✓ Sample scripts installed to ${INSTALL_DIR}/scripts/"
    echo "  Users can add custom scripts to ~/.serverdash/scripts/"
fi

# Auto-detect and set run_as_user for user-level tools
if [ -n "$SUDO_USER" ]; then
    echo "Auto-configuring run_as_user for: $SUDO_USER"
    sudo sed -i "s/\"run_as_user\": \"\"/\"run_as_user\": \"$SUDO_USER\"/" "${INSTALL_DIR}/config.json"
fi

# Install systemd service
sudo cp serverdash.service "${SERVICE_FILE}"
sudo systemctl daemon-reload
sudo systemctl enable serverdash

# Start the service
echo ""
echo "Starting ServerDash service..."
sudo systemctl start serverdash

# Wait a moment for service to start
sleep 2

# Show service status
if systemctl is-active --quiet serverdash; then
    echo "✓ ServerDash service is running"
else
    echo "⚠ ServerDash service may not have started properly"
    echo "  Check status with: sudo systemctl status serverdash"
fi

echo ""
echo "Installation complete!"
echo ""
echo "The web UI is available at:"
echo "  http://localhost:6432"
echo ""
echo "Useful commands:"
echo "  sudo systemctl status serverdash  # Check service status"
echo "  sudo systemctl restart serverdash # Restart service"
echo "  sudo journalctl -u serverdash -f   # View logs"
echo ""
echo "Sample scripts are installed in: ${INSTALL_DIR}/scripts/"
echo "Add custom scripts to: ~/.serverdash/scripts/"
echo ""
