#!/bin/bash
# WordPress Permissions Fixer
# Sets correct CHMOD/CHOWN for WordPress installations

set -e

# Get parameters from environment variables
WP_PATH="${path:-/var/www/html}"
ENVIRONMENT="${environment:-production}"

# Web server user (Ubuntu/Debian default)
WEB_USER="www-data"
WEB_GROUP="www-data"

# Get current user (for development environments)
CURRENT_USER="${SUDO_USER:-$USER}"

echo "=========================================="
echo "WordPress Permissions Fixer"
echo "=========================================="
echo "Path: $WP_PATH"
echo "Environment: $ENVIRONMENT"
echo "Web User: $WEB_USER:$WEB_GROUP"
echo ""

# Validate WordPress installation
if [[ ! -d "$WP_PATH" ]]; then
    echo "ERROR: Directory does not exist: $WP_PATH"
    exit 1
fi

if [[ ! -f "$WP_PATH/wp-config.php" ]]; then
    echo "ERROR: wp-config.php not found. Is this a WordPress installation?"
    exit 1
fi

echo "✓ WordPress installation detected"
echo ""

# Function to set permissions for production
set_production_permissions() {
    echo "Applying PRODUCTION permissions (stricter security)..."

    # Set ownership to web server user
    echo "→ Setting ownership to $WEB_USER:$WEB_GROUP"
    chown -R $WEB_USER:$WEB_GROUP "$WP_PATH"

    # Set directory permissions to 755
    echo "→ Setting directories to 755 (rwxr-xr-x)"
    find "$WP_PATH" -type d -exec chmod 755 {} \;

    # Set file permissions to 644
    echo "→ Setting files to 644 (rw-r--r--)"
    find "$WP_PATH" -type f -exec chmod 644 {} \;

    # wp-config.php should be more restrictive
    echo "→ Securing wp-config.php to 600 (rw-------)"
    chmod 600 "$WP_PATH/wp-config.php"

    # Ensure uploads directory is writable (for media uploads)
    if [[ -d "$WP_PATH/wp-content/uploads" ]]; then
        echo "→ Setting uploads directory to 775 (writable by web server)"
        find "$WP_PATH/wp-content/uploads" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/uploads" -type f -exec chmod 664 {} \;
    fi

    # Ensure cache directory is writable if it exists
    if [[ -d "$WP_PATH/wp-content/cache" ]]; then
        echo "→ Setting cache directory to 775"
        find "$WP_PATH/wp-content/cache" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/cache" -type f -exec chmod 664 {} \;
    fi

    # Ensure plugins/themes directories are writable for updates
    if [[ -d "$WP_PATH/wp-content/plugins" ]]; then
        echo "→ Setting plugins directory to 755 (writable by web server)"
        find "$WP_PATH/wp-content/plugins" -type d -exec chmod 755 {} \;
        find "$WP_PATH/wp-content/plugins" -type f -exec chmod 644 {} \;
    fi

    if [[ -d "$WP_PATH/wp-content/themes" ]]; then
        echo "→ Setting themes directory to 755"
        find "$WP_PATH/wp-content/themes" -type d -exec chmod 755 {} \;
        find "$WP_PATH/wp-content/themes" -type f -exec chmod 644 {} \;
    fi

    echo "✓ Production permissions applied"
}

# Function to set permissions for staging
set_staging_permissions() {
    echo "Applying STAGING permissions (balanced)..."

    # Set ownership to web server user
    echo "→ Setting ownership to $WEB_USER:$WEB_GROUP"
    chown -R $WEB_USER:$WEB_GROUP "$WP_PATH"

    # Set directory permissions to 755
    echo "→ Setting directories to 755 (rwxr-xr-x)"
    find "$WP_PATH" -type d -exec chmod 755 {} \;

    # Set file permissions to 644
    echo "→ Setting files to 644 (rw-r--r--)"
    find "$WP_PATH" -type f -exec chmod 644 {} \;

    # wp-config.php should be readable but protected
    echo "→ Setting wp-config.php to 640 (rw-r-----)"
    chmod 640 "$WP_PATH/wp-config.php"

    # Ensure uploads directory is writable
    if [[ -d "$WP_PATH/wp-content/uploads" ]]; then
        echo "→ Setting uploads directory to 775"
        find "$WP_PATH/wp-content/uploads" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/uploads" -type f -exec chmod 664 {} \;
    fi

    # Ensure cache directory is writable
    if [[ -d "$WP_PATH/wp-content/cache" ]]; then
        echo "→ Setting cache directory to 775"
        find "$WP_PATH/wp-content/cache" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/cache" -type f -exec chmod 664 {} \;
    fi

    # Plugins/themes should be writable for updates
    if [[ -d "$WP_PATH/wp-content/plugins" ]]; then
        find "$WP_PATH/wp-content/plugins" -type d -exec chmod 755 {} \;
        find "$WP_PATH/wp-content/plugins" -type f -exec chmod 644 {} \;
    fi

    if [[ -d "$WP_PATH/wp-content/themes" ]]; then
        find "$WP_PATH/wp-content/themes" -type d -exec chmod 755 {} \;
        find "$WP_PATH/wp-content/themes" -type f -exec chmod 644 {} \;
    fi

    echo "✓ Staging permissions applied"
}

# Function to set permissions for development
set_development_permissions() {
    echo "Applying DEVELOPMENT permissions (more permissive)..."

    # Check if current user is in www-data group
    if groups "$CURRENT_USER" | grep -q "$WEB_GROUP"; then
        echo "→ User $CURRENT_USER is already in $WEB_GROUP group"
        USER_IN_GROUP=true
    else
        echo "→ Adding $CURRENT_USER to $WEB_GROUP group..."
        usermod -aG "$WEB_GROUP" "$CURRENT_USER" 2>/dev/null || true
        USER_IN_GROUP=false
    fi

    # Set ownership to current user:www-data group
    # This allows:
    # - Your user to edit all files
    # - Web server to write to uploads/cache
    # - No need for sudo when editing files
    echo "→ Setting ownership to $CURRENT_USER:$WEB_GROUP"
    chown -R "$CURRENT_USER:$WEB_GROUP" "$WP_PATH"

    # Set directory permissions to 775 (user+group can write)
    echo "→ Setting directories to 775 (rwxrwxr-x)"
    find "$WP_PATH" -type d -exec chmod 775 {} \;

    # Set file permissions to 664 (user+group can write)
    echo "→ Setting files to 664 (rw-rw-r--)"
    find "$WP_PATH" -type f -exec chmod 664 {} \;

    # wp-config.php protection (still restrict access)
    echo "→ Setting wp-config.php to 640 (rw-r-----)"
    chmod 640 "$WP_PATH/wp-config.php"
    chown "$CURRENT_USER:$WEB_GROUP" "$WP_PATH/wp-config.php"

    # Ensure uploads/cache are writable by both user and web server
    if [[ -d "$WP_PATH/wp-content/uploads" ]]; then
        echo "→ Ensuring uploads directory is writable"
        find "$WP_PATH/wp-content/uploads" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/uploads" -type f -exec chmod 664 {} \;
    fi

    if [[ -d "$WP_PATH/wp-content/cache" ]]; then
        echo "→ Ensuring cache directory is writable"
        find "$WP_PATH/wp-content/cache" -type d -exec chmod 775 {} \;
        find "$WP_PATH/wp-content/cache" -type f -exec chmod 664 {} \;
    fi

    if [[ -d "$WP_PATH/wp-content/upgrade" ]]; then
        echo "→ Ensuring upgrade directory is writable"
        chmod 775 "$WP_PATH/wp-content/upgrade"
    fi

    echo "✓ Development permissions applied"
    echo ""
    echo "Development setup recommendations:"
    echo "  • Files owned by: $CURRENT_USER:$WEB_GROUP"
    echo "  • You can edit files without sudo"
    echo "  • Web server can write to uploads/cache"
    if [[ "$USER_IN_GROUP" != "true" ]]; then
        echo ""
        echo "⚠ NOTE: You may need to log out and back in for group changes to take effect."
        echo "  Or run: newgrp $WEB_GROUP"
    fi
}

# Apply permissions based on environment
case "$ENVIRONMENT" in
    production)
        set_production_permissions
        ;;
    staging)
        set_staging_permissions
        ;;
    development)
        set_development_permissions
        ;;
    *)
        echo "ERROR: Invalid environment '$ENVIRONMENT'. Use: production, staging, or development"
        exit 1
        ;;
esac

echo ""
echo "=========================================="
echo "Summary of permissions applied:"
echo "=========================================="
echo "Location: $WP_PATH"
echo "Environment: $ENVIRONMENT"
echo ""

# Show a sample of the applied permissions
echo "Sample permissions:"
if [[ -d "$WP_PATH/wp-content" ]]; then
    ls -la "$WP_PATH/" | head -10
fi

echo ""
echo "✓ WordPress permissions fixed successfully!"
echo "=========================================="
